#!/bin/bash
#
# Gets KODAK Scanner Log and Computer Information
# v1.2
#
# Information will be in kds_eklog.YYYYMMDDhhmmss.tgz

# save the stderr and stdout handles so we can put them back
exec 5>&1
exec 6>&2

# display please wait
echo ""
echo ""
echo ""
echo "************************"
echo "Getting KODAK Scanner log information, please wait..."
echo ""

# display message that user has no access and exit script
no_access()
{
	# put stderr and stdout back, now that we are done redirecting
	exec 1>&5
	exec 2>&6

	echo ""
	echo "*** ERROR ***"
	if [ "$1" != "" ]; then
		echo $1
	fi
	echo "You may not have write access in this directory."
	echo "Try running this script from a different directory."
	echo ""
	exit
}

# check to see if every command we need is there
check_command()
{
	whichResult=`which $1 2>> /dev/null`
	if [ "$whichResult" = "" ]; then
		echo ""
		echo "*** ERROR ***"
		echo "This script requires command:" $1
		echo ""
		exit
	fi
}

# clear file. as part of this, make sure user has write access
clear_file()
{
	# make sure there is no file so we are not fooled
	# into thinking the echo worked if user has no rights
	rm -fr $1
	if [ -f "$1" ]; then
		no_access
	fi

	# write something to the file to test write access
	echo "test" > $1
	if [ ! -f "$1" ]; then
		no_access
	fi

	# able to write file, now remove the test file
	rm -fr $1
}

# create a directory, but make sure it is empty
create_directory()
{
	# if the directory exists, then ask user to remove it
	if [ -d $1 ]; then
		echo ""
		echo "*** WARNING ***"
		echo "Directory" $1 "already exists"
		echo -n "Delete directory and continue? (y/N) "
		read kdsPrompt
		echo ""
		if [ "$kdsPrompt" != "y" ] && [ "$kdsPrompt" != "Y" ]; then
			echo ""
			echo "*** WARNING ***"
			echo "No log information created."
			echo "Run this script from a different directory"
			echo "    or remove directory" $1
			echo ""
			exit
		fi

		rm -fr $1	# user said to remove it

		# make sure it is gone
		if [ -d $1 ]; then
			no_access
		fi
	fi
	mkdir $1		# create the directory
	if [ ! -d $1 ]; then	# make sure directory is there
		no_access
	fi

	# make sure we can create files in the directory
	cd $1
	clear_file kds_test.txt	
	cd ..
}

# clear file. as part of this, make sure user has write access
run_cmd()
{
	echo ""
	echo "*** CMD:" $*
	$*
}

# set file names just once
KDS_LOG_NOW=`date "+%Y%m%d%H%M%S"`
KDS_LOG_FILENAME=kds_eklog
KDS_LOG_OUTPUT=$KDS_LOG_FILENAME.txt
KDS_LOG_TAR=$KDS_LOG_FILENAME.$KDS_LOG_NOW.tgz
KDS_LOG_TMPDIR=$KDS_LOG_FILENAME

# make sure all commands that we need are installed
check_command tar
check_command script

# make sure there is no log file, so we can replace it
clear_file $KDS_LOG_TAR

# create a temporary directory to do all of the work we need to
create_directory $KDS_LOG_TMPDIR
cd $KDS_LOG_TMPDIR

# Get device information
exec >$KDS_LOG_FILENAME.dev.txt 2>&1
lsusbpath=`which lsusb 2>> /dev/null`
if [ "$lsusbpath" != "" ]; then
	run_cmd lsusb
	run_cmd lsusb -v
else
	run_cmd echo "No lsusb"
fi
run_cmd ls -la -R /dev/bus/usb
lshalpath=`which lshal 2>> /dev/null`
if [ "$lshalpath" != "" ]; then
	run_cmd lshal
else
	run_cmd echo "No lshal"
fi

# Get file information
exec >$KDS_LOG_FILENAME.file.txt 2>&1
run_cmd ls -la -R /usr/local/lib
run_cmd ls -la /opt
run_cmd ls -la -R /opt/kodak
run_cmd ls -la /var
run_cmd ls -la -R /var/kodak
run_cmd ls -la -R /etc/sane.d
run_cmd ls -la /etc/udev/rules.d
run_cmd ls -la -R /usr/lib/sane
run_cmd ls -la -R /usr/local/etc
run_cmd ls -la -R /usr/local/bin
run_cmd which scanimage
run_cmd ls -la -R /opt/sitdf	# temporary for Caixa

# Get user information
exec >$KDS_LOG_FILENAME.user.txt 2>&1
run_cmd whoami
run_cmd echo $USER
run_cmd logname
run_cmd id -nu
run_cmd id -u
run_cmd id
run_cmd cat /etc/group

# Get environment information
exec >$KDS_LOG_FILENAME.env.txt 2>&1
run_cmd set

# Get process information
exec >$KDS_LOG_FILENAME.process.txt 2>&1
run_cmd ps aux
run_cmd ps axms
run_cmd vmstat -D
run_cmd top -n 1 -b

# Get PC information
exec >$KDS_LOG_FILENAME.pc.txt 2>&1
run_cmd uname -a
run_cmd lsb_release -a
run_cmd ls -la /etc/*_version
run_cmd cat /etc/*_version
run_cmd ls -la /etc/*-release
run_cmd cat /etc/*-release
run_cmd cat /proc/cpuinfo
run_cmd cat /proc/meminfo
run_cmd dmesg
run_cmd lspci
run_cmd dpkg -l

# tar the files
exec >$KDS_LOG_FILENAME.tar.txt 2>&1
KDS_FILES_TO_TAR=/var/kodak
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR /etc/sane.d/kds_*.conf"
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR /etc/udev/rules.d/55-kodakdi.rules"
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR /opt/kodak/kds_*/install/*"
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR /tmp/kds_*/install.log"
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR $TMPDIR/kds_*/install.log"
KDS_FILES_TO_TAR="$KDS_FILES_TO_TAR $KDS_LOG_FILENAME.*.txt"
run_cmd tar -czvf $KDS_LOG_TAR $KDS_FILES_TO_TAR
if [ ! -f "$KDS_LOG_TAR" ]; then
	no_access "tar file $KDS_LOG_TAR was not created"
fi

# put stderr and stdout back, now that we are done redirecting
exec 1>&5
exec 2>&6

# move up out of the temp directory, move tar and remove temp directory
cd ..
mv $KDS_LOG_TMPDIR/$KDS_LOG_TAR $KDS_LOG_TAR
if [ ! -f "$KDS_LOG_TAR" ]; then
	echo ""
	echo "*** WARNING ***"
	echo "Could not move file: $KDS_LOG_TMPDIR/$KDS_LOG_TAR"
	echo ""
	KDS_LOG_TAR=$KDS_LOG_TMPDIR/$KDS_LOG_TAR
else
	# removed temp directory
	rm -fr $KDS_LOG_TMPDIR
	if [ -d "$KDS_LOG_TMPDIR" ]; then
		echo ""
		echo "*** WARNING ***"
		echo "Could not delete directory: $KDS_LOG_TMPDIR"
		echo ""
	fi
fi

# print message that we are finished
echo ""
echo "*** FINISHED ***"
echo "All information is in file:" $KDS_LOG_TAR
echo ""

