#!/bin/bash
#
# Uninstalls Kodak i2000 Scanner
# v1.0

# initialize variables
ADF_WITH_FLATBED_ACC="i1300 i1400 i2000 i4000"	# NOTE: do NOT add i1200 to this list
UNINSTALL_MODEL=i2000
UNINSTALL_MODEL_DIR=kds_i2000
UNINSTALL_FB1_DIR=
UNINSTALL_FB1_PKG=
UNINSTALL_FB2_DIR=
UNINSTALL_FB2_PKG=

# remove the given directory if it is empty
function remove_empty_dir()
{
	# see if directory is even there
	if [ ! -d "$1" ]; then
		return
	fi

	# directory is there, so see "kds_" directories
	if [ "`ls $1 | grep kds_`" == "" ]; then
		rm -fr $1	# is 'empty' so remove it
		return
	fi
}

# remove/uninstall the given package
function remove_package()
{
	# prior to removing, make sure it is there first
	if [ "$dpkgpath" != "" ]; then
		if [ "`dpkg --get-selections $1 2>>/dev/null | grep ^$1`" != "" ]; then
			$dpkgpath -P $1
		fi
	else
		if [ "`$rpmpath -qa | grep $1-`" != "" ]; then
			$rpmpath -e --nodeps $1
		fi
	fi
}

# make sure the user is root
if [ `id -u` != "0" ]; then
	echo ""
	echo ""
	echo ""
	echo "*** ERROR ***"
	echo "You must be logged into the root account (i.e. super user)"
	echo ""
	exit
fi


# determine the uninstall program to use
dpkgpath=`which dpkg 2>> /dev/null`
if [ "$dpkgpath" == "" ]; then
	rpmpath=`which rpm 2>> /dev/null`
	if [ "$rpmpath" == "" ]; then
		echo ""
		echo ""
		echo ""
		echo "*** ERROR ***"
		echo "Package installer (e.g. rpm) is not installed. Unable to uninstall driver."
		echo ""
		exit
	fi
fi

# make sure the user provided the model to remove
# NOTE: handle a future ability to set the Model at top of this script
if [ "$UNINSTALL_MODEL" == "" ]; then
	if [ "$1" == "" ]; then
		echo ""
		echo ""
		echo ""
		echo "***ERROR ***"
		echo "You must include the model to remove (e.g. i2000)."
		echo ""
		echo "Usage:"
		echo "   $0 i2000"
		echo ""
		exit
	fi
	UNINSTALL_MODEL=$1
fi
if [ "$UNINSTALL_MODEL_DIR" == "" ]; then
	UNINSTALL_MODEL_DIR=kds_$UNINSTALL_MODEL
fi
if [ "$dpkgpath" != "" ]; then
	UNINSTALL_MODEL_PKG=kodak-$UNINSTALL_MODEL
else
	# we have a special case of Legal Flatbed, because it got a "_2" appended to the package, but it should not have
	if [ "$UNINSTALL_MODEL" == "i2000legal" ]; then
		UNINSTALL_MODEL_PKG=kodak_i2000legal_2
	else
		UNINSTALL_MODEL_PKG=kodak_$UNINSTALL_MODEL
	fi
fi

	

# determine if any Flatbeds were on the ADF's CD; assume there are none
# FIRST:  check for any ADFs that have integrated flatbeds as
#            they must always be removed when the ADF is
# SECOND: check for i1200 has it is a special case. it is the only ADF
#            that shipped with the old accessory flatbeds, but later
#            came with the new accesory flatbed drivers
# THRID:  check for any ADFs that had the accessory flatbeds
#
# NOTE:   since we only ever had the old Arch flatbeds with an i1200, we
#         do not have to worry that we might remove them from another product

# FIRST
if  [ "$UNINSTALL_MODEL" == "i2900" ]; then
	# has integrated flatbed, but it has naming issues :-(
	UNINSTALL_FB1_DIR=kds_i2900a4
	if [ "$dpkgpath" != "" ]; then
		UNINSTALL_FB1_PKG=kodak-i2900a4
	else
		UNINSTALL_FB1_PKG=kodak_i2900A4_2
	fi
fi

# SECOND
if  [ "$UNINSTALL_MODEL" == "i1200" ]; then
	# since we do not know if this ADF is the old arch or the new arch,
	# we need to try and figure it out. If the ADF is not even installed,
	# then we have no idea, so skip any flatbed stuff. If the ADF is
	# installed, then see if the old arch flatbed is installed. If the old
	# arch flatbed is installed, then remove it; otherwise assume new arch.
	if [ -d /opt/kodak/$UNINSTALL_MODEL_DIR ]; then
		if [ -d /opt/kodak/kds_i1000A3 ]; then
			UNINSTALL_FB1_DIR=kds_i1000A3
			UNINSTALL_FB2_DIR=kds_i1000A4
			if [ "$dpkgpath" != "" ]; then
				UNINSTALL_FB1_PKG=kodak-i1000a3
				UNINSTALL_FB2_PKG=kodak-i1000a4
			else
				UNINSTALL_FB1_PKG=kodak_i1000A3
				UNINSTALL_FB2_PKG=kodak_i1000A4
			fi
		else
			ADF_WITH_FLATBED_ACC="i1200 "$ADF_WITH_FLATBED_ACC
		fi
	fi
fi

# THRID
FLATBED_ACC=0
for uninstall_model in $ADF_WITH_FLATBED_ACC
do
	# if this is the model we are uninstalling, then remove its accessories
	if [ $uninstall_model == $UNINSTALL_MODEL ]; then
		FLATBED_ACC=1
	fi

	# if this is not the model being removed, then see if it is installed.
	# if it is installed, then there isat least one other ADF that needs
	# the accessory flatbeds. when that is the case, then we need can
	# just stop searching now and 'lie' and say there are not accessory
	# flatbeds for this ADF (this works even if the ADF doesn't have them)
	if [ $uninstall_model != $UNINSTALL_MODEL ]; then
		if [ "$dpkgpath" != "" ]; then
			if [ "`dpkg --get-selections kodak-$uninstall_model 2>>/dev/null | grep ^kodak-$uninstall_model`" != "" ]; then
				FLATBED_ACC=0
				break
			fi
		else
			if [ "`$rpmpath -qa | grep kodak_$uninstall_model-`" != "" ]; then
				FLATBED_ACC=0
				break
			fi
		fi
	fi
done
if [ "$FLATBED_ACC" != "0" ]; then
	UNINSTALL_FB1_DIR=kds_i1000a3_2
	UNINSTALL_FB2_DIR=kds_i1000a4_2
	if [ "$dpkgpath" != "" ]; then
		UNINSTALL_FB1_PKG=kodak-i1000a3-2
		UNINSTALL_FB2_PKG=kodak-i1000a4-2
	else
		UNINSTALL_FB1_PKG=kodak_i1000A3_2
		UNINSTALL_FB2_PKG=kodak_i1000A4_2
	fi
fi



# make sure the scanner is disconnected
echo ""
echo ""
echo ""
echo "****************************************************************************"
echo "Turn off all scanners and unplug all scanner USB cables from the PC"
echo ""
echo "Press Enter to continue..."
read kdsPrompt


# we can't be in /opt if we're going to be deleting its contents. So, save
# the current folder and change the working directory to 'home'.
_saved_dir=`pwd`
cd ~


# remove old software
echo ""
echo ""
echo ""
echo "****************************************************************************"
echo "Removing software, please wait..."
echo ""
for uninstall_pkg in $UNINSTALL_FB1_PKG $UNINSTALL_FB2_PKG $UNINSTALL_MODEL_PKG
do
	remove_package $uninstall_pkg
done



# remove scanner specific folders
echo ""
echo ""
echo ""
echo "****************************************************************************"
echo "Packages are removed; now removing files, please wait..."
echo ""
# remove symbolic link (should never be one, but did see it on one PC)
rm /var/kodak 2>>/dev/null

# remove model (i.e. ADF and Flatbed) specific files
for uninstall_dir in $UNINSTALL_FB1_DIR $UNINSTALL_FB2_DIR $UNINSTALL_MODEL_DIR
do
	rm -rf /var/kodak/$uninstall_dir
	rm -rf /opt/kodak/$uninstall_dir
done

# remove ADF driver specific files
rm -rf /usr/local/lib/twain/kodak/$UNINSTALL_MODEL_DIR
rm -f /etc/sane.d/$UNINSTALL_MODEL_DIR.conf
rm -f /usr/lib/sane/libsane-$UNINSTALL_MODEL_DIR.*

# remove any empty folder
remove_empty_dir "/var/kodak"
remove_empty_dir "/opt/kodak"



# if no more drivers installed, then remove all the shared stuff
if [ ! -d "/opt/kodak" ]; then
	echo ""
	echo ""
	echo ""
	echo "****************************************************************************"
	echo "Removed shared packages, please wait..."
	echo ""

	# no more drivers, so remove the rules
	rm -fr /etc/udev/rules.d/55-kodakdi.rules

	# removed shared packages
	for uninstall_pkg in scantwain libopenusb twaindsm
	do
		remove_package $uninstall_pkg
	done
fi


# if the folder we started in still exists, then return to it
cd "$_saved_dir" > /dev/null 2>&1

# if the starting folder has been deleted, tell the user
_new_dir=`pwd`
if [ "$_new_dir" != "$_saved_dir" ]; then
	echo ""
	echo ""
	echo ""
	echo "****************************************************************************"
	echo "The starting folder '$_saved_dir' has been removed."
	echo "Please cd to an existing folder..."
	echo ""
fi


echo ""
echo ""
echo ""
echo "****************************************************************************"
echo "Uninstall is complete"
echo ""

